﻿using System;
using System.Data.SqlClient;
using DicomObjects.Enums;
using System.IO;
using DicomObjects;
using System.Threading;
using DicomObjects.EventArguments;
using System.Linq;
using System.Windows.Forms;
using Microsoft.Win32;
using System.Text;

namespace DicomServerDemo
{
    public partial class Form1 : Form
    {
        MainServer Server;

        private const string MdfPath = @"C:\ProgramData\DicomServerDB.mdf";
        private const string OpenSubKeyPath = "SYSTEM\\CurrentControlSet\\services";
        private const string MsSql = "MSSQL$";
        private const string MsSqlServer = "MSSQLSERVER";
        private const string DbConnectionFailedMessage = "The server failed to attach the database to SQL server - do you have an accessible copy of SQL Server installed for which you have permission to add databases?\r\n\r\nThe errors were:\r\n";

        public Form1()
        {
            InitializeComponent();
        }

        private void Form1_Load(object sender, EventArgs e)
        {
            Server = new MainServer();
            Server.Log += Log;
            SqlConnection DBConnection = DatabaseConnection();
            if(DBConnection != null)
                Server.Startup(DBConnection);

            DicomGlobal.LogEvent += DicomGlobal_LogEvent; // for detailed logging
            ThreadPool.SetMaxThreads(5, 5);
        }

        private void Form1_FormClosing(object sender, FormClosingEventArgs e)
        {
            Server?.ShutDown();
        }

        // this routine is fairly generic - searching through available SQL server instances utnil it finds one which allows
        // us to attach the database.  Feel free to edit to something more specific for your environment
        SqlConnection DatabaseConnection()
        {
            var Errors = new StringBuilder();
            SqlConnection DBConnection = new SqlConnection();

            RegistryKey key = Registry.LocalMachine.OpenSubKey(OpenSubKeyPath);

            var sqlservers = from k in key.GetSubKeyNames() where k.StartsWith(MsSql) || k == MsSqlServer select k;

            foreach (string svr in sqlservers)
            {
                string servername = (svr == MsSqlServer) ? "." : ".\\" + svr.Substring(6);
                DBConnection.ConnectionString = $@"Server={servername};Database=master;Trusted_Connection=Yes;MultipleActiveResultSets=True;";

                try
                {
                    DBConnection.Open();
                    Log($"Opened database on {servername}");

                    // Check if DicomServerDB database exists
                    string checkDbQuery = "SELECT database_id FROM sys.databases WHERE Name = 'DicomServerDB'";
                    SqlCommand checkDbCmd = new SqlCommand(checkDbQuery, DBConnection);
                    var dbExists = checkDbCmd.ExecuteScalar();
                    DBConnection.Close();

                    if (dbExists == null)
                    {
                        // Database does not exist, attach the MDF file and rebuild the log
                        string attachDbQuery = $@"CREATE DATABASE DicomServerDB ON
                                                    (FILENAME = '{MdfPath}')
                                                    FOR ATTACH_REBUILD_LOG;";
                        DBConnection.Open();

                        SqlCommand attachCmd = new SqlCommand(attachDbQuery, DBConnection);
                        attachCmd.ExecuteNonQuery(); 
                        Log($"Attached database from {MdfPath} and rebuilt log on {servername}");

                        // Close the connection to master and connect to DicomServerDB
                        DBConnection.Close();
                        DBConnection.ConnectionString = $@"Server={servername};Database=DicomServerDB;Trusted_Connection=Yes;MultipleActiveResultSets=True;";
                        DBConnection.Open();
                    }
                    else
                    {
                        // Database exists, connect without attaching
                        DBConnection.ConnectionString = $@"Server={servername};Database=DicomServerDB;Trusted_Connection=Yes;MultipleActiveResultSets=True;";
                        DBConnection.Open();
                        Log($"Opened existing database DicomServerDB on {servername}");
                    }

                    return DBConnection;
                }
                catch (Exception ex) // record, then try next server
                {
                    if (DBConnection.State == System.Data.ConnectionState.Open)
                    {
                        DBConnection.Close();
                    }
                    Errors.Append($"\r\nDatabase connection to {servername} failed.  Error was {ex.Message}\r\n");
                }
            }
            Log(Errors.ToString());
            MessageBox.Show(DbConnectionFailedMessage + Errors.ToString());

            return null;
        }

        #region Logging Routines

        // DicomObjects native logging
        void DicomGlobal_LogEvent(LogEventArgs e)
        {
            if (LiveLogging.Checked)
            {
                AppendText(logger, e.Text);
            }
        }

        void AppendText(TextBox box, string text)
        {
            if (box.InvokeRequired)
            {
                BeginInvoke(new Action<TextBox, string>(AppendText), new object[] { box, text });
            }
            else
            {
                box.AppendText(text + Environment.NewLine);
            }
        }

        void Log(string s)
        {
            AppendText(logger, s);
        }

        private void LiveLogging_CheckedChanged(object sender, EventArgs e)
        {
            DicomGlobal.EventLogLevel = (LogLevel)(LiveLogging.Checked ? 63 : 0);
        }
        #endregion
    }
}
